#pragma once
#include <fstream>
#include <vector>
#include <set>
#include <unordered_set>

#include "HelperTypes.h"

class MeshLoader 
{
protected:
    std::vector<Node> nodes;
    std::vector<FiniteElement> elements;
    std::vector<BoundaryFiniteElement> boundary;

public:

    virtual void LoadMesh(const std::string&) = 0;
    virtual ~MeshLoader() = default;

    void Print_Data()
    {
        std::cout << Get_Nodes() << std::endl
            << Get_Elements() << std::endl
            << Get_Surfaces() << std::endl;
    }


    // ,    ,     
    std::vector<Node>& Get_Nodes()
    {
        return nodes;
    }
    std::vector<FiniteElement>& Get_Elements()
    {
        return elements;
    }
    std::vector<BoundaryFiniteElement>& Get_Surfaces()
    {
        return boundary;
    }

    // ,     ID    
    std::vector<FiniteElement> Get_Elements_by_ID(int id1, int id2, int id3)
    {
        std::vector<FiniteElement> answer;
        auto current_iter = elements.begin();
        while (current_iter != elements.end()) {
            current_iter = std::find_if(current_iter, elements.end(), [id1, id2, id3](const FiniteElement& current_element) {
                return std::find(current_element.nodes_id.begin(), current_element.nodes_id.end(), id1) != current_element.nodes_id.end()
                    && std::find(current_element.nodes_id.begin(), current_element.nodes_id.end(), id2) != current_element.nodes_id.end()
                    && std::find(current_element.nodes_id.begin(), current_element.nodes_id.end(), id3) != current_element.nodes_id.end(); });
            if (current_iter != elements.end()) {
                answer.push_back(*current_iter);
                ++current_iter;
            }
        }
        return answer;
    }


    // ,     
    std::vector<FiniteElement> Get_Elements_by_rib(int id1, int id2)
    {
        std::vector<FiniteElement> answer;

        auto current_iter = elements.begin();
        while (current_iter != elements.end()) 
        {
            current_iter = std::find_if(current_iter, elements.end(), [id1, id2](const FiniteElement& current_element) {
                return std::find(current_element.nodes_id.begin(), current_element.nodes_id.end(), id1) != current_element.nodes_id.end()
                    && std::find(current_element.nodes_id.begin(), current_element.nodes_id.end(), id2) != current_element.nodes_id.end(); });
            if (current_iter != elements.end()) {
                answer.push_back(*current_iter);
                ++current_iter;
            }
        }
        return answer;
    }



    // ,      ID  
    std::vector<FiniteElement> Get_Elements_by_ID_area(int a_id)
    {
        std::vector<FiniteElement> answer;
        auto current_iter = elements.begin();
        while (current_iter != elements.end())
        {
            current_iter = std::find_if(current_iter, elements.end(), [a_id](const FiniteElement& s) { return s.material_id == a_id; });
            if (current_iter != elements.end()) 
            {
                answer.push_back(*current_iter);
                ++current_iter;
            }
        }
        return answer;
    }


    // ,      ID 
    std::vector<Node> Get_Vertex_Nodes_by_Border_ID(int s_id)
    {
        std::vector<Node> answer;
        auto current_iter = boundary.begin();
        while (current_iter != boundary.end()) 
        {
            current_iter = std::find_if(current_iter, boundary.end(), [s_id](const BoundaryFiniteElement& s) { return s.border_id == s_id; });
            if (current_iter != boundary.end()) {
                for (const auto& it : current_iter->nodes_id) 
                {
                    auto node_by_ID = std::find_if(nodes.begin(), nodes.end(), [it](Node n) { return n.id == it; });
                    answer.push_back(*node_by_ID);
                }
                ++current_iter;
            }
        }
        return answer;
    }


    // ,       ID 
    std::vector<BoundaryFiniteElement> Get_Borders_by_Border_ID(int s_id)
    {
        std::vector<BoundaryFiniteElement> answer;
        auto surface_by_ID = boundary.begin();
        while (surface_by_ID != boundary.end()) 
        {
            surface_by_ID = std::find_if(surface_by_ID, boundary.end(), [s_id](const BoundaryFiniteElement& s) {
                return s.border_id == s_id; });
            if (surface_by_ID != boundary.end()) 
            {
                answer.push_back(*surface_by_ID);
                ++surface_by_ID;
            }
        }
        return answer;
    }

    void Insert_New_Nodes() 
    {
        std::unordered_set<Edge, EdgeHash> edge_container;
        for (auto& it : elements)
        {
            std::vector<int> current_iter(it.nodes_id);
            for (auto i = current_iter.begin(); i != current_iter.end() - 1; i++)
            {
                for (auto j = i + 1; j != current_iter.end(); j++)
                {
                    Edge current_edge(*i, *j);
                    auto search_edge = edge_container.find(current_edge);
                    Node left_node = Get_Node_by_ID(*i);
                    Node right_node = Get_Node_by_ID(*j);
                    Node center_node(nodes.size() + 1, (left_node.x + right_node.x) / 2,
                        (left_node.y + right_node.y) / 2, (left_node.z + right_node.z) / 2, false);

                    if (search_edge == edge_container.end())
                    {
                        current_edge.self_id = nodes.size() + 1;
                        edge_container.insert(current_edge);
                        nodes.push_back(center_node);
                        it.nodes_id.push_back(current_edge.self_id);
                    }
                    else
                    {
                        it.nodes_id.push_back(search_edge->self_id);
                    }

                }
            }
        }

        for (auto& it : boundary)
        {
            std::vector<int> current_iter(it.nodes_id);
            for (auto i = current_iter.begin(); i != current_iter.end() - 1; i++)
            {
                for (auto j = i + 1; j != current_iter.end(); j++)
                {
                    Edge search_edge(*i, *j);
                    auto search_point = edge_container.find(search_edge);
                    if (search_point != edge_container.end())
                    {
                        it.nodes_id.push_back(search_point->self_id);
                    }
                }
            }
        }
    }


    // ,  , n-         n
    std::vector<std::set<int>> Get_Nodes_Neighbors() 
    {
        std::vector<std::set<int>> result(nodes.size());
        for (const auto& it_FiniteElement : elements)
        {
            for (auto& it_nodes : it_FiniteElement.nodes_id)
            {
                result[it_nodes - 1].insert(it_FiniteElement.nodes_id.begin(), it_FiniteElement.nodes_id.end());
                result[it_nodes - 1].erase(it_nodes);
            }
        }

        return result;
    }

    Node Get_Node_by_ID(int n_id)
    {
        for (const auto& it : nodes)
            if (it.id == n_id)
                return it;
        return {};
    }

};


